<?php
/**
 * Plugin Name: EdgeOne Gutenberg Authentication (Universal)
 * Description: EdgeOne 鉴权 (方式A) + 强制尺寸控制。同时支持图片和视频区块，通过 Style 强制生效。
 * Version: 1.5
 * Author: Marukon
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// =================================================================
// 1. 后台设置 (密钥配置)
// =================================================================

function edgeone_register_settings() {
    register_setting( 'edgeone_auth_options', 'edgeone_secret_key' );
    add_settings_section( 'edgeone_main_section', '基础配置', null, 'edgeone-auth-settings' );
    add_settings_field( 'edgeone_secret_key', '鉴权密钥 (Secret Key)', 'edgeone_secret_key_callback', 'edgeone-auth-settings', 'edgeone_main_section' );
}
add_action( 'admin_init', 'edgeone_register_settings' );

function edgeone_secret_key_callback() {
    $value = get_option( 'edgeone_secret_key', '' );
    echo '<input type="text" name="edgeone_secret_key" value="' . esc_attr( $value ) . '" class="regular-text" />';
    echo '<p class="description">请输入腾讯云 EdgeOne 鉴权配置中的主密钥。</p>';
}

function edgeone_add_options_page() {
    add_options_page( 'EdgeOne 鉴权', 'EdgeOne Auth', 'manage_options', 'edgeone-auth-settings', 'edgeone_render_options_page' );
}
add_action( 'admin_menu', 'edgeone_add_options_page' );

function edgeone_render_options_page() {
    ?>
    <div class="wrap"><h1>EdgeOne 鉴权设置</h1><form action="options.php" method="post"><?php settings_fields( 'edgeone_auth_options' ); do_settings_sections( 'edgeone-auth-settings' ); submit_button(); ?></form></div>
    <?php
}

// =================================================================
// 2. 古腾堡编辑器 (JS - 图片和视频通用控件)
// =================================================================

function edgeone_enqueue_block_editor_assets() {
    $script = <<<JS
    (function(wp) {
        var addFilter = wp.hooks.addFilter;
        var el = wp.element.createElement;
        var InspectorControls = wp.blockEditor.InspectorControls;
        var PanelBody = wp.components.PanelBody;
        var ToggleControl = wp.components.ToggleControl;
        var TextControl = wp.components.TextControl;
        var createHigherOrderComponent = wp.compose.createHigherOrderComponent;

        // 1. 注册通用属性
        function addEdgeOneAttributes(settings, name) {
            // 目标：核心图片区块 和 核心视频区块
            if (name === 'core/image' || name === 'core/video') {
                if (!settings.attributes) settings.attributes = {};
                
                // 鉴权开关
                settings.attributes.edgeoneAuth = { type: 'boolean', default: false };
                
                // 尺寸控制 (通用)
                settings.attributes.edgeoneWidth = { type: 'string', default: '' };
                settings.attributes.edgeoneHeight = { type: 'string', default: '' };
            }
            return settings;
        }
        addFilter('blocks.registerBlockType', 'edgeone/add-attributes', addEdgeOneAttributes);

        // 2. 侧边栏控件
        var withEdgeOneControls = createHigherOrderComponent(function(BlockEdit) {
            return function(props) {
                var name = props.name;
                // 仅在图片和视频区块显示
                if (name !== 'core/image' && name !== 'core/video') return el(BlockEdit, props);

                var attributes = props.attributes;
                var setAttributes = props.setAttributes;

                return el(
                    wp.element.Fragment, {},
                    el(BlockEdit, props),
                    el(InspectorControls, {}, 
                        el(PanelBody, { title: 'EdgeOne & 尺寸设置', initialOpen: true }, 
                            // 鉴权开关
                            el(ToggleControl, {
                                label: '启用 EdgeOne 鉴权',
                                checked: attributes.edgeoneAuth || false,
                                help: '前端自动生成签名 URL',
                                onChange: function(val) { setAttributes({ edgeoneAuth: val }); }
                            }),
                            
                            el('hr', {}),
                            
                            // 尺寸说明
                            el('p', { style: {fontSize:'12px', color:'#666', marginBottom: '10px'} }, 
                               '强制尺寸 (覆盖主题 CSS)'
                            ),
                            
                            // 宽度输入
                            el(TextControl, {
                                label: '宽度 (Width)',
                                value: attributes.edgeoneWidth || '',
                                placeholder: '如: 500 或 100%',
                                help: '自动补齐 px',
                                onChange: function(val) { setAttributes({ edgeoneWidth: val }); }
                            }),
                            
                            // 高度输入
                            el(TextControl, {
                                label: '高度 (Height)',
                                value: attributes.edgeoneHeight || '',
                                placeholder: '如: 300 或 auto',
                                onChange: function(val) { setAttributes({ edgeoneHeight: val }); }
                            })
                        )
                    )
                );
            };
        }, 'withEdgeOneControls');

        addFilter('editor.BlockEdit', 'edgeone/with-inspector-controls', withEdgeOneControls);
    })(window.wp);
JS;
    wp_register_script('edgeone-block-script', '', ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components'], '1.5', true);
    wp_add_inline_script('edgeone-block-script', $script);
    wp_enqueue_script('edgeone-block-script');
}
add_action( 'enqueue_block_editor_assets', 'edgeone_enqueue_block_editor_assets' );

// =================================================================
// 3. PHP 核心逻辑 (鉴权 + 样式注入)
// =================================================================

function edgeone_sign_url( $url ) {
    $key = get_option( 'edgeone_secret_key' );
    if ( empty( $key ) || empty( $url ) ) return $url;

    $p = parse_url( $url );
    if ( ! isset( $p['path'] ) ) return $url;
    
    $path = $p['path'];
    $ts = time();
    $rand = substr( str_shuffle( '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ' ), 0, 16 );
    $uid = 0;
    
    $sign = md5( "$path-$ts-$rand-$uid-$key" );
    $token = "$ts-$rand-$uid-$sign";
    
    $scheme = isset($p['scheme']) ? $p['scheme'] . '://' : '//';
    $host = isset($p['host']) ? $p['host'] : '';
    $query = isset($p['query']) ? $p['query'] : '';
    $query = $query ? "$query&token=$token" : "token=$token";
    
    return "$scheme$host$path?$query";
}

function edgeone_render_filter( $content, $block ) {
    // 检查区块类型：图片 或 视频
    if ( $block['blockName'] !== 'core/image' && $block['blockName'] !== 'core/video' ) return $content;
    
    // 依赖 WP 6.2+
    if ( ! class_exists( 'WP_HTML_Tag_Processor' ) ) return $content;

    $tags = new WP_HTML_Tag_Processor( $content );
    // 图片找 <img>，视频找 <video>
    $tag_name = ($block['blockName'] === 'core/image') ? 'img' : 'video';
    
    if ( $tags->next_tag( array( 'tag_name' => $tag_name ) ) ) {
        
        // --- 1. 处理强制尺寸 ---
        // 获取属性 (使用通用变量名 edgeoneWidth / edgeoneHeight)
        $w = isset($block['attrs']['edgeoneWidth']) ? trim($block['attrs']['edgeoneWidth']) : '';
        $h = isset($block['attrs']['edgeoneHeight']) ? trim($block['attrs']['edgeoneHeight']) : '';

        if ( ! empty( $w ) || ! empty( $h ) ) {
            $style_parts = array();
            
            // 宽度处理
            if ( ! empty( $w ) ) {
                $tags->set_attribute( 'width', $w ); 
                $w_css = is_numeric( $w ) ? $w . 'px' : $w;
                $style_parts[] = "width: {$w_css}";
            }

            // 高度处理
            if ( ! empty( $h ) ) {
                $tags->set_attribute( 'height', $h );
                $h_css = is_numeric( $h ) ? $h . 'px' : $h;
                $style_parts[] = "height: {$h_css}";
            }

            // 写入内联 Style
            if ( ! empty( $style_parts ) ) {
                // 注意：这里会覆盖标签原有的 style 属性。
                // 如果图片本身有旋转等其他 style，可能会丢失。但通常 core/image 依赖 class 控制样式。
                // 强制覆盖是解决“无效”最直接的方法。
                $new_style = implode( '; ', $style_parts ) . ';';
                $tags->set_attribute( 'style', $new_style );
            }
        }

        // --- 2. 处理鉴权 ---
        if ( ! empty( $block['attrs']['edgeoneAuth'] ) ) {
            $src = $tags->get_attribute( 'src' );
            if ( $src ) {
                $tags->set_attribute( 'src', edgeone_sign_url( $src ) );
            }
        }

        return $tags->get_updated_html();
    }

    return $content;
}
add_filter( 'render_block', 'edgeone_render_filter', 10, 2 );